/*:
 * @target MZ
 * @plugindesc v1.0.1 現在のキー/ゲームパッド/マウス割当を一覧表示する「キーガイド」。他プラグインの推定ホットキーも拾います。
 * @author Human Simulate (HS)
 *
 * @help HS_KeyGuide.js
 * 
 * ■ できること
 *  - 今このゲームで有効なキー割当を自動収集して一覧表示
 *  - Keyboard / Gamepad / Mouse（推定）/ Plugin推定ホットキー をカテゴリ別に表示
 *  - オプションに「キーガイド」項目を追加（ON/OFF可、タイトルで非表示可）
 *  - プラグインコマンドからも呼び出し可
 *  - シンボル（ok/cancel/shift/backlog/skip等）→説明文を言語別にカスタム可
 * 
 * ■ 注意
 *  - Input.keyMapper / gamepadMapper に登録されていない“生の keyCode を直接読む”型は自動検出できない場合があります。
 *    その場合は ActionLabels で説明文を補ってください。
 *  - 「他プラグインの推定ホットキー」は $plugins のパラメータ文字列をヒューリスティックに走査します。
 *    表示は参考値です（確定ではありません）。
 * 
 * @param ShowInOptions
 * @text オプションに項目を追加
 * @type boolean
 * @default true
 *
 * @param ShowInOptionsOnTitle
 * @text タイトルのオプションにも表示
 * @type boolean
 * @default false
 *
 * @param OptionName
 * @text オプション項目名
 * @type string
 * @default キーガイド / Key Guide
 *
 * @param LanguageVarId
 * @text 言語切替 変数ID（0=日本語固定）
 * @type variable
 * @default 0
 * @desc 1=日本語, 2=English, 3=简体中文, 4=한국어 を想定。0なら日本語固定。
 *
 * @param ActionLabels
 * @text シンボル→説明(既定)
 * @type struct<ActionLabel>[]
 * @default [
 * {"symbol":"ok","labelJp":"決定 / 進む","labelEn":"OK / Advance","labelZh":"确认 / 前进","labelKo":"확인 / 진행"},
 * {"symbol":"cancel","labelJp":"キャンセル / メニューを閉じる","labelEn":"Cancel / Close","labelZh":"取消 / 关闭","labelKo":"취소 / 닫기"},
 * {"symbol":"menu","labelJp":"メニューを開く","labelEn":"Open Menu","labelZh":"打开菜单","labelKo":"메뉴 열기"},
 * {"symbol":"shift","labelJp":"ダッシュ / （作品依存：スキップ等）","labelEn":"Dash / (project-specific: skip)","labelZh":"冲刺 /（依项目而定：快进）","labelKo":"대시 / (작품 의존: 스킵)"},
 * {"symbol":"pageup","labelJp":"ページアップ /（作品依存：前ログ）","labelEn":"Page Up / (project-specific: prev log)","labelZh":"上一页 /（依项目而定）","labelKo":"페이지 업 / (작품 의존)"},
 * {"symbol":"pagedown","labelJp":"ページダウン /（作品依存：次ログ）","labelEn":"Page Down / (project-specific: next log)","labelZh":"下一页 /（依项目而定）","labelKo":"페이지 다운 / (작품 의존)"},
 * {"symbol":"up","labelJp":"カーソル上","labelEn":"Cursor Up","labelZh":"上","labelKo":"위"},
 * {"symbol":"down","labelJp":"カーソル下","labelEn":"Cursor Down","labelZh":"下","labelKo":"아래"},
 * {"symbol":"left","labelJp":"カーソル左","labelEn":"Cursor Left","labelZh":"左","labelKo":"왼쪽"},
 * {"symbol":"right","labelJp":"カーソル右","labelEn":"Cursor Right","labelZh":"右","labelKo":"오른쪽"}
 * ]
 *
 * @command ShowKeyGuide
 * @text キーガイドを開く
 * @desc 一覧シーンを開きます。
 *
 * @command CopyJson
 * @text JSONをクリップボードへ
 * @desc 現在の割当をJSONでコピー（NW.js環境推奨）
 */

/*~struct~ActionLabel:
 * @param symbol
 * @text シンボル名
 * @type string
 * @desc Input.keyMapper の値（ok/cancel/shift等）や独自シンボル（skip/backlog等）
 *
 * @param labelJp
 * @text 説明(日本語)
 * @type string
 * @default
 *
 * @param labelEn
 * @text 説明(English)
 * @type string
 * @default
 *
 * @param labelZh
 * @text 説明(简体中文)
 * @type string
 * @default
 *
 * @param labelKo
 * @text 説明(한국어)
 * @type string
 * @default
 */

(() => {
  const PLUGIN_NAME = "HS_KeyGuide";
  const params = PluginManager.parameters(PLUGIN_NAME);
  const pBool = (k, d=false)=> String(params[k] ?? d) === "true";
  const pStr  = (k, d="")   => String(params[k] ?? d);
  const pVar  = (k, d=0)    => Number(params[k] ?? d);
  const pJson = (k, d="[]") => { try { return JSON.parse(params[k] ?? d); } catch(e){ return JSON.parse(d); } };

  const SHOW_IN_OPTIONS        = pBool("ShowInOptions", true);
  const SHOW_IN_OPTIONS_TITLE  = pBool("ShowInOptionsOnTitle", false);
  const OPTION_NAME            = pStr("OptionName", "キーガイド / Key Guide");
  const LANG_VAR_ID            = pVar("LanguageVarId", 0);
  const ACTION_LABELS          = pJson("ActionLabels");

  // 言語インデックス（1=JP,2=EN,3=ZH,4=KO）
  function currentLangIndex(){
    if (LANG_VAR_ID > 0 && typeof $gameVariables !== "undefined" && $gameVariables) {
      const v = Number($gameVariables.value(LANG_VAR_ID) || 1);
      return Math.max(1, Math.min(4, v|0));
    }
    return 1; // JP固定
  }

  // シンボル→説明辞書
  const labelDict = {};
  for (const it of ACTION_LABELS) {
    const sym = (it.symbol || "").trim();
    if (!sym) continue;
    labelDict[sym] = [it.labelJp||"", it.labelEn||"", it.labelZh||"", it.labelKo||""];
  }
  function labelFor(symbol){
    const lang = currentLangIndex();
    const entry = labelDict[symbol];
    if (entry && entry[lang-1]) return entry[lang-1];
    const fallback = {
      ok:"決定 / 進む",
      cancel:"キャンセル",
      menu:"メニュー",
      shift:"シフト",
      pageup:"ページアップ",
      pagedown:"ページダウン",
      up:"カーソル上", down:"カーソル下", left:"カーソル左", right:"カーソル右",
      escape:"キャンセル", control:"コントロール"
    };
    return fallback[symbol] || "";
  }

  // キーコード→名称
  function keyNameFromCode(code){
    const table = {
      8:"Backspace",9:"Tab",13:"Enter",16:"Shift",17:"Ctrl",18:"Alt",
      19:"Pause",20:"CapsLock",27:"Esc",32:"Space",
      33:"PageUp",34:"PageDown",35:"End",36:"Home",
      37:"Left",38:"Up",39:"Right",40:"Down",
      45:"Insert",46:"Delete",
      91:"Meta",93:"Menu",
    };
    if (table[code]) return table[code];
    if (code>=65 && code<=90) return String.fromCharCode(code); // A-Z
    if (code>=48 && code<=57) return String.fromCharCode(code); // 0-9
    if (code>=96 && code<=105) return "Numpad " + (code-96);
    if (code>=112 && code<=123) return "F" + (code-111);
    return "KeyCode " + code;
  }

  // Gamepadボタン名（一般的なXInput準拠）
  function gamepadButtonName(i){
    const map = {
      0:"A (0)",1:"B (1)",2:"X (2)",3:"Y (3)",
      4:"LB (4)",5:"RB (5)",6:"LT (6)",7:"RT (7)",
      8:"Back (8)",9:"Start (9)",10:"L3 (10)",11:"R3 (11)",
      12:"D-Pad Up (12)",13:"D-Pad Down (13)",14:"D-Pad Left (14)",15:"D-Pad Right (15)"
    };
    return map[i] ?? ("Button " + i);
  }

  // 現在の一覧データ生成
  function buildGuideData(){
    const data = {
      keyboard: [],
      gamepad:  [],
      mouse:    [],
      pluginGuessed: []
    };

    // Keyboard
    const km = Input.keyMapper || {};
    const entries = Object.entries(km).map(([codeStr, symbol])=>{
      const code = Number(codeStr);
      return {
        keyName: keyNameFromCode(code),
        code, symbol: String(symbol),
        label: labelFor(String(symbol))
      };
    }).sort((a,b)=>{
      if (a.symbol===b.symbol) return a.code - b.code;
      return a.symbol.localeCompare(b.symbol);
    });
    data.keyboard = entries;

    // Gamepad
    const gm = Input.gamepadMapper || {};
    const gp = Object.entries(gm).map(([btnStr, symbol])=>{
      const idx = Number(btnStr);
      return {
        buttonName: gamepadButtonName(idx),
        index: idx, symbol: String(symbol),
        label: labelFor(String(symbol))
      };
    }).sort((a,b)=>{
      if (a.symbol===b.symbol) return a.index - b.index;
      return a.symbol.localeCompare(b.symbol);
    });
    data.gamepad = gp;

    // Mouse / Touch（MZ標準の挙動からの“定番”推定）
    data.mouse = [
      {mouse:"左クリック", symbol:"ok",     label:labelFor("ok")},
      {mouse:"右クリック/長押し(キャンセル相当)", symbol:"cancel", label:labelFor("cancel")},
      {mouse:"ホイール上/下（作品依存）", symbol:"pageup/pagedown", label:"スクロール/ログ移動など（作品依存）"}
    ];

    // 他プラグインのパラメータからキーっぽい単語を推定
    try {
      const KEY_WORDS = /(hot\s*key|shortcut|key\s*code|key|キー|ショートカット|ホットキー)/i;
      const LOOKS_LIKE_KEY = /^(?:[A-Z]|[0-9]|F[1-9][0-2]?|Space|Enter|Esc|Escape|Shift|Ctrl|Alt|Tab|Page(?:Up|Down)|Home|End|Insert|Delete|Left|Right|Up|Down)$/i;
      if (Array.isArray($plugins)) {
        for (const pl of $plugins) {
          if (!pl || !pl.parameters) continue;
          const pluginName = pl.name || "(unknown)";
          for (const [k,v] of Object.entries(pl.parameters)) {
            if (!k) continue;
            const val = String(v).trim();
            if (!val) continue;
            const keyLike = KEY_WORDS.test(k) || LOOKS_LIKE_KEY.test(val);
            if (!keyLike) continue;
            data.pluginGuessed.push({plugin: pluginName, param: k, value: val});
          }
        }
      }
    } catch(e){}

    return data;
  }

  // ============ Scene / Window ============
  class Window_KeyGuideList extends Window_Selectable {
    initialize(rect){
      super.initialize(rect);
      this._items = [];
      this.refreshData();
      this.activate();
    }
    maxItems(){ return this._items.length; }
    refreshData(){
      const d = buildGuideData();
      this._items = [];
      const pushHeader = (t)=> this._items.push({type:"header", text:t});
      const pushRow = (row)=> this._items.push(Object.assign({type:"row"}, row));

      const lang = currentLangIndex();
      const H = {
        kb: ["【キーボード】","[Keyboard]","【键盘】","【키보드】"][lang-1],
        gp: ["【ゲームパッド】","[Gamepad]","【手柄】","【게임패드】"][lang-1],
        ms: ["【マウス/タッチ（推定）】","[Mouse/Touch (assumed)]","【鼠标/触控（推测）】","【마우스/터치(추정)】"][lang-1],
        pg: ["【プラグインの“推定”ホットキー】","[Plugins' guessed hotkeys]","【插件“推测”热键】","【플러그인 추정 핫키】"][lang-1]
      };
      const SRC = {
        kb: "Input.keyMapper",
        gp: "Input.gamepadMapper",
        ms: ["TouchInput(推定)","TouchInput (assumed)","TouchInput（推测）","TouchInput(추정)"][lang-1]
      };

      // キーボード
      pushHeader(H.kb);
      for (const it of d.keyboard) {
        pushRow({col1: `${it.keyName}`, col2: it.symbol, col3: it.label, col4: SRC.kb});
      }
      // ゲームパッド
      pushHeader(H.gp);
      for (const it of d.gamepad) {
        pushRow({col1: `${it.buttonName}`, col2: it.symbol, col3: it.label, col4: SRC.gp});
      }
      // マウス推定
      pushHeader(H.ms);
      for (const it of d.mouse) {
        pushRow({col1: it.mouse, col2: it.symbol, col3: it.label, col4: SRC.ms});
      }
      // プラグイン推定
      pushHeader(H.pg);
      for (const it of d.pluginGuessed) {
        pushRow({col1: it.value, col2: "-", col3: `param: ${it.param}`, col4: it.plugin});
      }
      this.refresh();
    }
    // ★ MZ互換：ColorManager.textDimColor 非使用（不透明度で薄表示）
    drawItem(index){
      const rect = this.itemRect(index);
      const item = this._items[index];
      this.resetTextColor();
      if (item.type === "header"){
        this.changeTextColor(ColorManager.systemColor());
        this.drawText(item.text, rect.x, rect.y, rect.width);
        this.changePaintOpacity(true);
      } else {
        const w = rect.width;
        const colW = Math.floor(w/4);
        this.drawText(item.col1||"", rect.x+0*colW, rect.y, colW-6);
        this.drawText(item.col2||"", rect.x+1*colW, rect.y, colW-6);
        this.drawText(item.col3||"", rect.x+2*colW, rect.y, colW-6);
        this.changePaintOpacity(false);
        this.drawText(item.col4||"", rect.x+3*colW, rect.y, colW-6);
        this.changePaintOpacity(true);
        this.resetTextColor();
      }
    }
  }

  class Scene_KeyGuide extends Scene_MenuBase {
    create(){
      super.create();
      const margin = 12;
      const listRect = new Rectangle(
        margin,
        margin,
        Graphics.boxWidth  - margin * 2,
        Graphics.boxHeight - margin * 2
      );
      this._list = new Window_KeyGuideList(listRect);
      this.addWindow(this._list);

      // ヘルプ文（多言語）
      const lang = currentLangIndex();
      const help = [
        "現在の割当を自動取得。説明文は「シンボル→説明(既定)」で編集できます。",
        "Auto-detected bindings. Edit descriptions via plugin parameter ActionLabels.",
        "自动获取当前映射。说明可在参数中按符号编辑。",
        "현재 매핑 자동 취득. 설명은 파라미터에서 심볼별 편집."
      ][lang-1];

      // ★ MZ方式：Window_Help は Rect を渡す
      const helpH = (Window_Base.prototype.fittingHeight
        ? Window_Base.prototype.fittingHeight.call(this._list, 1)
        : this._list.lineHeight() + this._list.itemPadding()*2 + 24);
      const helpRect = new Rectangle(
        margin,
        Graphics.boxHeight - helpH - margin,
        Graphics.boxWidth  - margin * 2,
        helpH
      );
      this._helpWindow = new Window_Help(helpRect);
      this._helpWindow.setText(help);
      this.addWindow(this._helpWindow);

      this._list.height = Graphics.boxHeight - helpH - margin * 2;
    }
  }

  // プラグインコマンド
  PluginManager.registerCommand(PLUGIN_NAME, "ShowKeyGuide", () => {
    SceneManager.push(Scene_KeyGuide);
  });

  PluginManager.registerCommand(PLUGIN_NAME, "CopyJson", async () => {
    const d = buildGuideData();
    const text = JSON.stringify(d, null, 2);
    try {
      if (navigator && navigator.clipboard && navigator.clipboard.writeText){
        await navigator.clipboard.writeText(text);
        SoundManager.playOk();
      } else if (Utils.isNwjs && Utils.isNwjs()) {
        const fs = require("fs");
        const path = require("path");
        const out = path.join(process.cwd(), "HS_KeyGuide.json");
        fs.writeFileSync(out, text, "utf8");
        SoundManager.playOk();
      } else {
        console.warn("[HS_KeyGuide] クリップボードAPIが使えません。");
        SoundManager.playBuzzer();
      }
    } catch(e){
      console.error(e);
      SoundManager.playBuzzer();
    }
  });

  // オプションに項目追加
  if (SHOW_IN_OPTIONS) {
    const _Window_Options_makeCommandList = Window_Options.prototype.makeCommandList;
    Window_Options.prototype.makeCommandList = function(){
      _Window_Options_makeCommandList.call(this);
      // タイトルのオプションで非表示
      if (!SHOW_IN_OPTIONS_TITLE) {
        const isTitle =
          SceneManager._fromTitleForOptions === true ||
          (SceneManager._scene instanceof Scene_Options && SceneManager._scene._fromTitle);
        if (isTitle) return; // 他項目は既に base が作成済み
      }
      this.addCommand(OPTION_NAME, "__hs_keyguide__");
    };

    const _Window_Options_processOk = Window_Options.prototype.processOk;
    Window_Options.prototype.processOk = function(){
      const symbol = this.commandSymbol(this.index());
      if (symbol === "__hs_keyguide__"){
        SoundManager.playOk();
        SceneManager.push(Scene_KeyGuide);
        return;
      }
      _Window_Options_processOk.call(this);
    };

    // タイトルから来たかの簡易フラグ
    const _Scene_Title_commandOptions = Scene_Title.prototype.commandOptions;
    Scene_Title.prototype.commandOptions = function(){
      SceneManager._fromTitleForOptions = true;
      _Scene_Title_commandOptions.call(this);
    };
    const _Scene_Options_start = Scene_Options.prototype.start;
    Scene_Options.prototype.start = function(){
      this._fromTitle = !!SceneManager._fromTitleForOptions;
      SceneManager._fromTitleForOptions = false;
      _Scene_Options_start.call(this);
    };
  }

})();

